<?php
session_start();
if (!isset($_SESSION['admin_logged'])) {
    header('Location: login.php');
    exit;
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Licenças - IPTV Player Admin</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif; background: #0f0f0f; color: #fff; min-height: 100vh; }
        .header { background: #1a1a1a; padding: 20px; display: flex; justify-content: space-between; align-items: center; border-bottom: 1px solid #333; }
        .header h1 { color: #e50914; font-size: 24px; }
        .header nav a { color: #888; text-decoration: none; margin-left: 20px; }
        .header nav a:hover, .header nav a.active { color: #fff; }
        .container { max-width: 1200px; margin: 0 auto; padding: 20px; }
        .card { background: #1a1a1a; border-radius: 12px; padding: 24px; margin-bottom: 20px; }
        .card h2 { margin-bottom: 20px; }
        .form-row { display: flex; gap: 16px; align-items: flex-end; }
        .form-group { flex: 1; }
        .form-group label { display: block; margin-bottom: 8px; color: #aaa; }
        .form-group input { width: 100%; padding: 12px; border-radius: 8px; border: 1px solid #333; background: #252525; color: #fff; }
        .btn { padding: 12px 24px; border-radius: 8px; border: none; cursor: pointer; font-weight: 600; }
        .btn-primary { background: #e50914; color: #fff; }
        .btn-primary:hover { background: #f40612; }
        .btn-sm { padding: 6px 12px; font-size: 12px; }
        .btn-danger { background: #dc3545; color: #fff; }
        .btn-warning { background: #ffc107; color: #000; }
        .btn-success { background: #28a745; color: #fff; }
        .btn-info { background: #17a2b8; color: #fff; }
        table { width: 100%; border-collapse: collapse; }
        th, td { padding: 12px; text-align: left; border-bottom: 1px solid #333; }
        th { color: #888; font-weight: 500; }
        .badge { padding: 4px 8px; border-radius: 4px; font-size: 11px; font-weight: 600; }
        .badge-success { background: #1a472a; color: #4ade80; }
        .badge-danger { background: #4a1a1a; color: #f87171; }
        .badge-warning { background: #4a3a1a; color: #fbbf24; }
        .license-key { font-family: monospace; background: #252525; padding: 4px 8px; border-radius: 4px; }
        .alert { padding: 12px; border-radius: 8px; margin-bottom: 16px; }
        .alert-success { background: #1a472a; color: #4ade80; }
        .stats { display: grid; grid-template-columns: repeat(4, 1fr); gap: 16px; margin-bottom: 20px; }
        .stat { background: #1a1a1a; border-radius: 12px; padding: 20px; text-align: center; }
        .stat .number { font-size: 32px; font-weight: bold; color: #e50914; }
        .stat .label { color: #888; margin-top: 4px; }
        .actions { display: flex; gap: 8px; }
    </style>
</head>
<body>
    <div class="header">
        <h1>📺 IPTV Player Admin</h1>
        <nav>
            <a href="index.php">Playlists</a>
            <a href="licenses.php" class="active">Licenças</a>
            <a href="logout.php">Sair</a>
        </nav>
    </div>

    <div class="container">
        <div class="stats">
            <div class="stat">
                <div class="number" id="totalLicenses">0</div>
                <div class="label">Total Licenças</div>
            </div>
            <div class="stat">
                <div class="number" id="activeLicenses">0</div>
                <div class="label">Ativas</div>
            </div>
            <div class="stat">
                <div class="number" id="usedLicenses">0</div>
                <div class="label">Em Uso</div>
            </div>
            <div class="stat">
                <div class="number" id="availableLicenses">0</div>
                <div class="label">Disponíveis</div>
            </div>
        </div>

        <div class="card">
            <h2>Gerar Nova Licença</h2>
            <div id="alert"></div>
            <div class="form-row">
                <div class="form-group">
                    <label>Data de Expiração (opcional)</label>
                    <input type="date" id="expiresAt">
                </div>
                <button class="btn btn-primary" onclick="createLicense()">Gerar Licença</button>
            </div>
        </div>

        <div class="card">
            <h2>Licenças Cadastradas</h2>
            <table>
                <thead>
                    <tr>
                        <th>Chave</th>
                        <th>Status</th>
                        <th>Dispositivo</th>
                        <th>Expira em</th>
                        <th>Criada em</th>
                        <th>Ações</th>
                    </tr>
                </thead>
                <tbody id="licensesTable"></tbody>
            </table>
        </div>
    </div>

    <script>
        const API = 'api.php';

        async function loadLicenses() {
            const res = await fetch(`${API}?action=licenses`);
            const licenses = await res.json();
            
            const total = licenses.length;
            const active = licenses.filter(l => l.status === 'active').length;
            const used = licenses.filter(l => l.device_id).length;
            const available = licenses.filter(l => l.status === 'active' && !l.device_id).length;

            document.getElementById('totalLicenses').textContent = total;
            document.getElementById('activeLicenses').textContent = active;
            document.getElementById('usedLicenses').textContent = used;
            document.getElementById('availableLicenses').textContent = available;

            const tbody = document.getElementById('licensesTable');
            if (licenses.length === 0) {
                tbody.innerHTML = '<tr><td colspan="6" style="text-align:center;color:#666">Nenhuma licença cadastrada</td></tr>';
                return;
            }

            tbody.innerHTML = licenses.map(l => `
                <tr>
                    <td><span class="license-key">${l.license_key}</span></td>
                    <td>
                        <span class="badge ${l.status === 'active' ? 'badge-success' : l.status === 'expired' ? 'badge-warning' : 'badge-danger'}">
                            ${l.status === 'active' ? 'Ativa' : l.status === 'expired' ? 'Expirada' : 'Inativa'}
                        </span>
                    </td>
                    <td>${l.device_name || '<span style="color:#666">Não ativada</span>'}</td>
                    <td>${l.expires_at ? new Date(l.expires_at).toLocaleDateString('pt-BR') : 'Sem limite'}</td>
                    <td>${new Date(l.created_at).toLocaleDateString('pt-BR')}</td>
                    <td class="actions">
                        <button class="btn btn-sm ${l.status === 'active' ? 'btn-warning' : 'btn-success'}" onclick="toggleLicense(${l.id})">
                            ${l.status === 'active' ? 'Desativar' : 'Ativar'}
                        </button>
                        ${l.device_id ? `<button class="btn btn-sm btn-info" onclick="resetLicense(${l.id})">Reset</button>` : ''}
                        <button class="btn btn-sm btn-danger" onclick="deleteLicense(${l.id})">Excluir</button>
                    </td>
                </tr>
            `).join('');
        }

        async function createLicense() {
            const expiresAt = document.getElementById('expiresAt').value || null;
            const alert = document.getElementById('alert');

            const res = await fetch(`${API}?action=create_license`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ expires_at: expiresAt })
            });

            const data = await res.json();
            if (data.success) {
                alert.className = 'alert alert-success';
                alert.innerHTML = `✅ Licença criada: <strong>${data.license_key}</strong>`;
                document.getElementById('expiresAt').value = '';
                loadLicenses();
            }
        }

        async function toggleLicense(id) {
            await fetch(`${API}?action=toggle_license&id=${id}`);
            loadLicenses();
        }

        async function resetLicense(id) {
            if (!confirm('Resetar dispositivo desta licença?')) return;
            await fetch(`${API}?action=reset_license&id=${id}`);
            loadLicenses();
        }

        async function deleteLicense(id) {
            if (!confirm('Excluir esta licença?')) return;
            await fetch(`${API}?action=delete_license&id=${id}`);
            loadLicenses();
        }

        loadLicenses();
    </script>
</body>
</html>
