<?php
session_start();

// Verificar login
if (!isset($_SESSION['admin_logged'])) {
    header('Location: login.php');
    exit;
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Painel Admin - IPTV Player</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
            background: #0f0f0f;
            color: #fff;
            min-height: 100vh;
        }
        .header {
            background: #1a1a1a;
            padding: 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            border-bottom: 1px solid #333;
        }
        .header h1 { color: #e50914; font-size: 24px; }
        .header a { color: #888; text-decoration: none; }
        .header a:hover { color: #fff; }
        .container { max-width: 1200px; margin: 0 auto; padding: 20px; }
        .stats {
            display: grid;
            grid-template-columns: repeat(3, 1fr);
            gap: 20px;
            margin-bottom: 30px;
        }
        .stat-card {
            background: #1a1a1a;
            border-radius: 12px;
            padding: 24px;
            text-align: center;
        }
        .stat-card .number { font-size: 42px; font-weight: bold; color: #e50914; }
        .stat-card .label { color: #888; margin-top: 8px; }
        .card {
            background: #1a1a1a;
            border-radius: 12px;
            padding: 24px;
            margin-bottom: 20px;
        }
        .card h2 { margin-bottom: 20px; color: #fff; }
        .form-row { display: flex; gap: 16px; margin-bottom: 16px; }
        .form-group { flex: 1; }
        .form-group label { display: block; margin-bottom: 8px; color: #aaa; }
        .form-group input {
            width: 100%;
            padding: 12px 16px;
            border-radius: 8px;
            border: 1px solid #333;
            background: #252525;
            color: #fff;
            font-size: 14px;
        }
        .form-group input:focus { outline: none; border-color: #e50914; }
        .btn {
            padding: 12px 24px;
            border-radius: 8px;
            border: none;
            cursor: pointer;
            font-weight: 600;
            font-size: 14px;
            transition: all 0.2s;
        }
        .btn-primary { background: #e50914; color: #fff; }
        .btn-primary:hover { background: #f40612; }
        .btn-primary:disabled { background: #666; cursor: not-allowed; }
        .btn-danger { background: #dc3545; color: #fff; padding: 8px 16px; }
        .btn-danger:hover { background: #c82333; }
        .alert {
            padding: 16px;
            border-radius: 8px;
            margin-bottom: 16px;
            display: none;
        }
        .alert-success { background: #1a472a; color: #4ade80; display: block; }
        .alert-error { background: #4a1a1a; color: #f87171; display: block; }
        .alert-loading { background: #1a3a4a; color: #60a5fa; display: block; }
        table {
            width: 100%;
            border-collapse: collapse;
        }
        th, td {
            padding: 16px;
            text-align: left;
            border-bottom: 1px solid #333;
        }
        th { color: #888; font-weight: 500; }
        .empty { text-align: center; color: #666; padding: 40px; }
        @media (max-width: 768px) {
            .stats { grid-template-columns: 1fr; }
            .form-row { flex-direction: column; }
        }
    </style>
</head>
<body>
    <div class="header">
        <h1>📺 Painel Admin - IPTV Player</h1>
        <nav>
            <a href="index.php" class="active">Playlists</a>
            <a href="licenses.php">Licenças</a>
            <a href="logout.php">Sair</a>
        </nav>
    </div>

    <div class="container">
        <div class="stats">
            <div class="stat-card">
                <div class="number" id="totalPlaylists">-</div>
                <div class="label">Playlists</div>
            </div>
            <div class="stat-card">
                <div class="number" id="totalChannels">-</div>
                <div class="label">Canais</div>
            </div>
            <div class="stat-card">
                <div class="number" id="totalGroups">-</div>
                <div class="label">Categorias</div>
            </div>
        </div>

        <div class="card">
            <h2>Adicionar Nova Playlist</h2>
            <div id="alert" class="alert"></div>
            <div class="form-row">
                <div class="form-group">
                    <label>Nome da Playlist</label>
                    <input type="text" id="playlistName" placeholder="Ex: Lista Principal">
                </div>
                <div class="form-group">
                    <label>URL da Playlist (M3U/M3U8)</label>
                    <input type="url" id="playlistUrl" placeholder="http://exemplo.com/playlist.m3u">
                </div>
            </div>
            <button class="btn btn-primary" id="addBtn" onclick="addPlaylist()">
                Adicionar Playlist
            </button>
        </div>

        <div class="card">
            <h2>Playlists Cadastradas</h2>
            <div id="playlistsTable"></div>
        </div>
    </div>

    <script>
        const API = 'api.php';

        async function loadStats() {
            const res = await fetch(`${API}?action=stats`);
            const data = await res.json();
            document.getElementById('totalPlaylists').textContent = data.playlists;
            document.getElementById('totalChannels').textContent = data.channels;
            document.getElementById('totalGroups').textContent = data.groups;
        }

        async function loadPlaylists() {
            const res = await fetch(`${API}?action=playlists`);
            const playlists = await res.json();
            const container = document.getElementById('playlistsTable');

            if (playlists.length === 0) {
                container.innerHTML = '<p class="empty">Nenhuma playlist cadastrada</p>';
                return;
            }

            container.innerHTML = `
                <table>
                    <thead>
                        <tr>
                            <th>Nome</th>
                            <th>Canais</th>
                            <th>Data</th>
                            <th>Ações</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${playlists.map(p => `
                            <tr>
                                <td>${p.name}</td>
                                <td>${p.total_channels}</td>
                                <td>${new Date(p.created_at).toLocaleDateString('pt-BR')}</td>
                                <td>
                                    <button class="btn btn-danger" onclick="deletePlaylist(${p.id})">
                                        Remover
                                    </button>
                                </td>
                            </tr>
                        `).join('')}
                    </tbody>
                </table>
            `;
        }

        async function addPlaylist() {
            const name = document.getElementById('playlistName').value.trim();
            const url = document.getElementById('playlistUrl').value.trim();
            const alert = document.getElementById('alert');
            const btn = document.getElementById('addBtn');

            if (!name || !url) {
                alert.className = 'alert alert-error';
                alert.textContent = 'Preencha todos os campos';
                return;
            }

            btn.disabled = true;
            btn.textContent = 'Processando...';
            alert.className = 'alert alert-loading';
            alert.textContent = '⏳ Baixando e processando playlist... Isso pode levar alguns minutos.';

            try {
                const res = await fetch(`${API}?action=add_playlist`, {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ name, url })
                });

                const data = await res.json();

                if (data.error) throw new Error(data.error);

                alert.className = 'alert alert-success';
                alert.textContent = `✅ Playlist adicionada com ${data.total_channels} canais!`;
                document.getElementById('playlistName').value = '';
                document.getElementById('playlistUrl').value = '';
                loadPlaylists();
                loadStats();
            } catch (err) {
                alert.className = 'alert alert-error';
                alert.textContent = '❌ Erro: ' + err.message;
            } finally {
                btn.disabled = false;
                btn.textContent = 'Adicionar Playlist';
            }
        }

        async function deletePlaylist(id) {
            if (!confirm('Remover esta playlist e todos os canais?')) return;
            await fetch(`${API}?action=delete_playlist&id=${id}`);
            loadPlaylists();
            loadStats();
        }

        loadStats();
        loadPlaylists();
    </script>
</body>
</html>
