<?php
require_once 'config.php';
cors();

$db = getDB();
$action = $_GET['action'] ?? '';

// Parser M3U
function parseM3U($content) {
    $lines = explode("\n", $content);
    $channels = [];
    $current = null;
    
    foreach ($lines as $line) {
        $line = trim($line);
        
        if (strpos($line, '#EXTINF:') === 0) {
            preg_match('/,(.+)$/', $line, $nameMatch);
            preg_match('/tvg-logo="([^"]+)"/', $line, $logoMatch);
            preg_match('/group-title="([^"]+)"/', $line, $groupMatch);
            
            $current = [
                'name' => $nameMatch[1] ?? 'Canal',
                'logo' => $logoMatch[1] ?? '',
                'group_name' => $groupMatch[1] ?? 'Geral'
            ];
        } elseif ($line && strpos($line, '#') !== 0 && $current) {
            $current['url'] = $line;
            $channels[] = $current;
            $current = null;
        }
    }
    
    return $channels;
}

// Buscar conteúdo de URL
function fetchUrl($url) {
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_TIMEOUT => 120,
        CURLOPT_SSL_VERIFYPEER => false
    ]);
    $content = curl_exec($ch);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) throw new Exception($error);
    return $content;
}

switch ($action) {
    // Listar todas as playlists
    case 'playlists':
        $stmt = $db->query("SELECT p.*, COUNT(c.id) as total_channels 
                           FROM playlists p 
                           LEFT JOIN channels c ON c.playlist_id = p.id 
                           GROUP BY p.id 
                           ORDER BY p.created_at DESC");
        jsonResponse($stmt->fetchAll());
        break;
    
    // Adicionar playlist
    case 'add_playlist':
        $data = json_decode(file_get_contents('php://input'), true);
        $name = $data['name'] ?? '';
        $url = $data['url'] ?? '';
        
        if (!$name || !$url) {
            jsonResponse(['error' => 'Nome e URL são obrigatórios'], 400);
        }
        
        try {
            // Baixar e parsear playlist
            $content = fetchUrl($url);
            $channels = parseM3U($content);
            
            if (empty($channels)) {
                jsonResponse(['error' => 'Nenhum canal encontrado na playlist'], 400);
            }
            
            // Inserir playlist
            $stmt = $db->prepare("INSERT INTO playlists (name, url) VALUES (?, ?)");
            $stmt->execute([$name, $url]);
            $playlistId = $db->lastInsertId();
            
            // Inserir canais
            $stmt = $db->prepare("INSERT INTO channels (playlist_id, name, logo, group_name, url) VALUES (?, ?, ?, ?, ?)");
            foreach ($channels as $ch) {
                $stmt->execute([$playlistId, $ch['name'], $ch['logo'], $ch['group_name'], $ch['url']]);
            }
            
            jsonResponse([
                'success' => true,
                'playlist_id' => $playlistId,
                'total_channels' => count($channels)
            ]);
        } catch (Exception $e) {
            jsonResponse(['error' => 'Erro ao processar: ' . $e->getMessage()], 500);
        }
        break;
    
    // Deletar playlist
    case 'delete_playlist':
        $id = $_GET['id'] ?? 0;
        $stmt = $db->prepare("DELETE FROM playlists WHERE id = ?");
        $stmt->execute([$id]);
        jsonResponse(['success' => true]);
        break;
    
    // Listar todos os canais (para o app React)
    case 'channels':
        $stmt = $db->query("SELECT c.id, c.name, c.logo, c.group_name as `group`, c.url 
                           FROM channels c 
                           ORDER BY c.group_name, c.name");
        jsonResponse($stmt->fetchAll());
        break;
    
    // Estatísticas
    case 'stats':
        $playlists = $db->query("SELECT COUNT(*) FROM playlists")->fetchColumn();
        $channels = $db->query("SELECT COUNT(*) FROM channels")->fetchColumn();
        $groups = $db->query("SELECT COUNT(DISTINCT group_name) FROM channels")->fetchColumn();
        $licenses = $db->query("SELECT COUNT(*) FROM licenses")->fetchColumn();
        $activeLicenses = $db->query("SELECT COUNT(*) FROM licenses WHERE status = 'active'")->fetchColumn();
        jsonResponse([
            'playlists' => (int)$playlists,
            'channels' => (int)$channels,
            'groups' => (int)$groups,
            'licenses' => (int)$licenses,
            'active_licenses' => (int)$activeLicenses
        ]);
        break;
    
    // ========== SISTEMA DE LICENÇAS ==========
    
    // Verificar licença (chamado pelo app)
    case 'verify_license':
        $data = json_decode(file_get_contents('php://input'), true);
        $licenseKey = $data['license_key'] ?? '';
        $deviceId = $data['device_id'] ?? '';
        $deviceName = $data['device_name'] ?? 'Dispositivo';
        
        if (!$licenseKey || !$deviceId) {
            jsonResponse(['valid' => false, 'error' => 'Licença e ID do dispositivo são obrigatórios'], 400);
        }
        
        $stmt = $db->prepare("SELECT * FROM licenses WHERE license_key = ?");
        $stmt->execute([$licenseKey]);
        $license = $stmt->fetch();
        
        if (!$license) {
            jsonResponse(['valid' => false, 'error' => 'Licença não encontrada']);
        }
        
        if ($license['status'] !== 'active') {
            jsonResponse(['valid' => false, 'error' => 'Licença inativa ou expirada']);
        }
        
        if ($license['expires_at'] && strtotime($license['expires_at']) < time()) {
            $db->prepare("UPDATE licenses SET status = 'expired' WHERE id = ?")->execute([$license['id']]);
            jsonResponse(['valid' => false, 'error' => 'Licença expirada']);
        }
        
        // Se já tem device_id vinculado, verifica se é o mesmo
        if ($license['device_id'] && $license['device_id'] !== $deviceId) {
            jsonResponse(['valid' => false, 'error' => 'Licença já ativada em outro dispositivo']);
        }
        
        // Ativar licença no dispositivo
        if (!$license['device_id']) {
            $stmt = $db->prepare("UPDATE licenses SET device_id = ?, device_name = ?, activated_at = NOW() WHERE id = ?");
            $stmt->execute([$deviceId, $deviceName, $license['id']]);
        }
        
        jsonResponse([
            'valid' => true,
            'expires_at' => $license['expires_at'],
            'message' => 'Licença válida'
        ]);
        break;
    
    // Listar licenças (painel admin)
    case 'licenses':
        $stmt = $db->query("SELECT * FROM licenses ORDER BY created_at DESC");
        jsonResponse($stmt->fetchAll());
        break;
    
    // Criar nova licença (painel admin)
    case 'create_license':
        $data = json_decode(file_get_contents('php://input'), true);
        $expiresAt = $data['expires_at'] ?? null;
        
        // Gerar chave única
        $key = strtoupper(substr(md5(uniqid(rand(), true)), 0, 4) . '-' . 
               substr(md5(uniqid(rand(), true)), 0, 4) . '-' . 
               substr(md5(uniqid(rand(), true)), 0, 4) . '-' . 
               substr(md5(uniqid(rand(), true)), 0, 4));
        
        $stmt = $db->prepare("INSERT INTO licenses (license_key, expires_at) VALUES (?, ?)");
        $stmt->execute([$key, $expiresAt]);
        
        jsonResponse([
            'success' => true,
            'license_key' => $key,
            'id' => $db->lastInsertId()
        ]);
        break;
    
    // Deletar licença
    case 'delete_license':
        $id = $_GET['id'] ?? 0;
        $stmt = $db->prepare("DELETE FROM licenses WHERE id = ?");
        $stmt->execute([$id]);
        jsonResponse(['success' => true]);
        break;
    
    // Resetar dispositivo da licença
    case 'reset_license':
        $id = $_GET['id'] ?? 0;
        $stmt = $db->prepare("UPDATE licenses SET device_id = NULL, device_name = NULL, activated_at = NULL WHERE id = ?");
        $stmt->execute([$id]);
        jsonResponse(['success' => true]);
        break;
    
    // Alterar status da licença
    case 'toggle_license':
        $id = $_GET['id'] ?? 0;
        $stmt = $db->prepare("SELECT status FROM licenses WHERE id = ?");
        $stmt->execute([$id]);
        $current = $stmt->fetchColumn();
        $newStatus = $current === 'active' ? 'inactive' : 'active';
        $stmt = $db->prepare("UPDATE licenses SET status = ? WHERE id = ?");
        $stmt->execute([$newStatus, $id]);
        jsonResponse(['success' => true, 'status' => $newStatus]);
        break;
    
    default:
        jsonResponse(['error' => 'Ação não encontrada'], 404);
}
